﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using RevisionAnalyser.Projects;
using System.IO;
using System.Xml;
using RevisionAnalyser.Global;

namespace RevisionAnalyser
{
    public partial class NewProjectForm : Form
    {
        private string _file = string.Empty;
        private string _type = string.Empty;

        public NewProjectForm()
        {
            InitializeComponent();
        }

        private void CreateProject(string name, string location)
        {
            _file = String.Format("{0}\\{1}.raproj", location, name);
            try
            {
                XmlWriter xmlWriter = XmlWriter.Create(_file);
                try
                {
                    xmlWriter.WriteStartDocument();
                    xmlWriter.WriteStartElement("RAProject");
                    xmlWriter.WriteStartElement("GlobalSettings");
                    xmlWriter.WriteStartElement("ProjectType");
                    xmlWriter.WriteString(_type);
                    xmlWriter.WriteEndElement();
                    xmlWriter.WriteEndElement();
                    xmlWriter.WriteEndElement();
                    xmlWriter.WriteEndDocument();
                }
                finally
                {
                    xmlWriter.Close();
                }
                DialogResult = DialogResult.OK;
                Close();
            }
            catch
            {
                MessageBox.Show("Revision Analyser was unable to create the project.\n\nDo you have sufficient permissions?", "Could not create project", MessageBoxButtons.OK, MessageBoxIcon.Warning);
            }
        }

        public string GetProjectFile()
        {
            return _file;
        }

        private void EnableMode(bool mode)
        {
            uxNameField.Enabled = mode;
            uxLocationField.Enabled = mode;
            uxBrowseButton.Enabled = mode;
            uxOKButton.Enabled = mode;
            if (mode == false)
            {
                uxTypeNameLabel.Text = "Select a project";
                uxTypeDescriptionLabel.Text = "Please select a project type from the list.";
            }
        }

        private void NewProjectForm_Load(object sender, EventArgs e)
        {
            foreach (ProjectType projectType in ProjectTypes.Instance.GetProjectTypes())
            {
                uxProjectImages.Images.Add(projectType.Name, projectType.Icon);
                uxTypeView.Items.Add(new ListViewItem(projectType.Name, projectType.Name));
            }
            if (uxTypeView.Items.Count > 0)
            {
                uxTypeView.Items[0].Selected = true;
            }
            else
            {
                EnableMode(false);
            }
        }

        private void uxTypeView_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (uxTypeView.SelectedItems.Count > 0)
            {
                EnableMode(true);
                ProjectType projectType = ProjectTypes.Instance.GetProjectType(uxTypeView.SelectedItems[0].Text);
                if (projectType != null)
                {
                    _type = uxTypeView.SelectedItems[0].Text;
                    uxTypeNameLabel.Text = projectType.Name;
                    uxTypeDescriptionLabel.Text = projectType.Description;
                    uxNameField.Text = String.Format("{0}Project", projectType.Name);
                    uxLocationField.Text = String.Format("{0}\\{1}", Constants.DEFAULT_PROJECTS_PATH, uxNameField.Text);
                }
            }
            else
            {
                EnableMode(false);
            }
        }

        private void uxBrowseButton_Click(object sender, EventArgs e)
        {
            if (uxFolderBrowser.ShowDialog() == DialogResult.OK)
            {
                uxLocationField.Text = String.Format("{0}\\{1}", uxFolderBrowser.SelectedPath, uxNameField.Text);
            }
        }

        private void uxLocationField_TextChanged(object sender, EventArgs e)
        {
            try
            {
                string parent = Directory.GetParent(uxLocationField.Text).ToString();
                if (Directory.Exists(parent))
                {
                    uxFolderBrowser.SelectedPath = parent;
                }
                else
                {
                    uxFolderBrowser.SelectedPath = Constants.MY_DOCUMENTS_PATH;
                }
            }
            catch
            {
                uxFolderBrowser.SelectedPath = Constants.MY_DOCUMENTS_PATH;
            }
        }

        private void uxNameField_TextChanged(object sender, EventArgs e)
        {
            try
            {
                string parent = Directory.GetParent(uxLocationField.Text).ToString();
                uxLocationField.Text = String.Format("{0}\\{1}", parent, uxNameField.Text);
            }
            catch
            {
            }
        }

        private void uxOKButton_Click(object sender, EventArgs e)
        {
            if (uxNameField.Text.Trim().Length == 0)
            {
                MessageBox.Show("Please specify a project name.", "No project name", MessageBoxButtons.OK, MessageBoxIcon.Warning);
            }
            else if (Directory.Exists(uxLocationField.Text))
            {
                if (Directory.EnumerateFileSystemEntries(uxLocationField.Text).Any())
                {
                    MessageBox.Show("The selected folder is not empty.\n\nPlease remove all files in the folder and try again.", "Not an empty folder", MessageBoxButtons.OK, MessageBoxIcon.Warning);
                }
                else
                {
                    // Create project in empty folder.
                    CreateProject(uxNameField.Text, uxLocationField.Text);
                }
            }
            else
            {
                try
                {
                    // Try to create a new folder.
                    Directory.CreateDirectory(uxLocationField.Text);
                    CreateProject(uxNameField.Text, uxLocationField.Text);
                }
                catch
                {
                    MessageBox.Show("Revision Analyser was unable to create the selected folder.\n\nDo you have sufficient permissions?", "Could not create folder", MessageBoxButtons.OK, MessageBoxIcon.Warning);
                }
            }
        }

        private void uxNameField_KeyPress(object sender, KeyPressEventArgs e)
        {
            if (!((Char.IsLetter(e.KeyChar) || Char.IsNumber(e.KeyChar)) || Char.IsSeparator(e.KeyChar) || Char.IsControl(e.KeyChar)))
            {
                e.Handled = true;
            }
        }
    }
}
